import { Component, OnInit, OnDestroy, ElementRef, ViewChild, Renderer2 } from '@angular/core';
import { ActivatedRoute } from '@angular/router';
import { EventService } from '../store/services/event.service';
import { Store, select } from '@ngrx/store';
import { IAppState } from '../store/reducers/video.reducer';
import { FindTranscriptById } from '../store/actions/video.actions';
import { Observable, Subject } from 'rxjs';
import { takeUntil } from 'rxjs/operators';

@Component({
  selector: 'app-video-details',
  templateUrl: './video-details.component.html',
  styleUrls: ['./video-details.component.scss']
})
export class VideoDetailsComponent implements OnInit, OnDestroy {
  @ViewChild('player') private player: ElementRef;
  @ViewChild('video') private video: ElementRef;

  isDestroyed$ = new Subject<boolean>();
  videoIdParam: string;
  playing = false;
  videoLoaded = false;
  errorLoading = false;
  transcript$: Observable<any>;
  videoId$: Observable<string>;
  loading$: Observable<boolean>;
  error$: Observable<boolean>;
  private isMouseMoving = false;
  private events;

  constructor(
    private route: ActivatedRoute,
    private renderer: Renderer2,
    private evt: EventService,
    private store: Store<IAppState>
  ) {

    this.transcript$ = store.pipe(select(state => state.videoState.transcript));
    this.videoId$ = store.pipe(select(state => state.videoState.videoId));
    this.loading$ = store.pipe(select(state => state.videoState.loading));
    this.error$ = store.pipe(select(state => state.videoState.error));
  }

  ngOnInit() {
    this.route.queryParams.pipe(takeUntil(this.isDestroyed$))
      .subscribe(queryParams => {
        if (queryParams.id) {
          this.store.dispatch(new FindTranscriptById(queryParams.id)); 
        }
      });
      this.loading$.pipe(takeUntil(this.isDestroyed$))
      .subscribe(loading => {
        if (this.events) {
          this.evt.removeEvents(this.events);
        }
        this.load();
      })
  }

  ngOnDestroy() {
    this.evt.removeEvents(this.events);
    this.isDestroyed$.next(true);
    this.isDestroyed$.complete();
  }

  trackByFn(index, item) {
    return index;
  }

  load() {
    if (this.video && this.video.nativeElement) {
      this.video.nativeElement.load();

      this.events = [
        { element: this.video.nativeElement, name: 'loadstart', callback: event => this.videoLoaded = false, dispose: null },
        { element: this.video.nativeElement, name: 'loadedmetadata', callback: event => this.onLoadedMetadata(event), dispose: null },
        { element: this.video.nativeElement, name: 'error', callback: event => this.onLoadingError(event), dispose: null },
        { element: this.player.nativeElement, name: 'mouseenter', callback: event => this.onMouseEnter(event), dispose: null },
        { element: this.player.nativeElement, name: 'mouseleave', callback: event => this.onMouseLeave(event), dispose: null }
      ];

      this.evt.addEvents(this.renderer, this.events);
    }
  }

  onLoadingError(event) {
    this.errorLoading = true;
    console.error('Loading Error', event);
  }

  onLoadedMetadata(event: any) {
    this.videoLoaded = true;
  }

  onMouseEnter(event: any) {
    this.isMouseMoving = true;
  }

  onMouseLeave(event: any) {
    this.isMouseMoving = false;
  }

  toggleVideoPlayback(video) {
    this.playing = !this.playing;
    this.updateVideoPlayback(video);
  }

  updateVideoPlayback(video) {
    this.playing ? video.play() : video.pause();
  }

  showPlayButton(activeClass: string, inactiveClass: string): any {
    return (!this.playing || this.isMouseMoving) ? activeClass : inactiveClass;
  }

}
